<?php
if (!defined('ABSPATH')) {
    exit;
}

class CryptNote_Admin
{
    private string $option_key = 'cryptnote_pro_settings';
    private string $page_slug = 'cryptnote-pro';

    public function __construct()
    {
        add_action('admin_menu', [$this, 'register_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_post_cryptnote_pro_generate', [$this, 'handle_generate']);
        add_action('admin_notices', [$this, 'render_notices']);
    }

    public function register_menu(): void
    {
        add_menu_page(
            __('CryptNote', 'cryptnote-secure-links'),
            __('CryptNote', 'cryptnote-secure-links'),
            'manage_options',
            $this->page_slug,
            [$this, 'render_page'],
            'dashicons-lock',
            81
        );
    }

    public function register_settings(): void
    {
        register_setting(
            'cryptnote_pro_settings_group',
            $this->option_key,
            [
                'sanitize_callback' => [$this, 'sanitize_settings'],
                'default' => cryptnote_pro_default_settings(),
            ]
        );
    }

    public function sanitize_settings($input): array
    {
        $defaults = cryptnote_pro_default_settings();
        $input = is_array($input) ? $input : [];

        $api_base = isset($input['api_base']) ? esc_url_raw(trim((string)$input['api_base'])) : $defaults['api_base'];
        if ($api_base === '') {
            $api_base = $defaults['api_base'];
        }

        $max_views = isset($input['default_max_views']) ? (int)$input['default_max_views'] : $defaults['default_max_views'];
        $max_views = max(1, min(20, $max_views));

        $expire_input = isset($input['default_expire_minutes']) ? trim((string)$input['default_expire_minutes']) : '';
        $expire_minutes = $expire_input === '' ? '' : max(1, min(10080, (int)$expire_input));

        $format = isset($input['default_format']) ? sanitize_text_field($input['default_format']) : $defaults['default_format'];
        if (!in_array($format, ['text', 'markdown', 'html'], true)) {
            $format = 'text';
        }

        $clean = [
            'api_base' => rtrim($api_base, '/'),
            'api_key' => sanitize_text_field($input['api_key'] ?? ''),
            'default_max_views' => $max_views,
            'default_expire_minutes' => $expire_minutes,
            'default_format' => $format,
            'default_markdown' => $format === 'markdown' ? 1 : 0,
            'email_enable' => !empty($input['email_enable']) ? 1 : 0,
            'email_template' => sanitize_textarea_field($input['email_template'] ?? $defaults['email_template']),
            'email_send_html' => !empty($input['email_send_html']) ? 1 : 0,
            'email_template_html' => wp_kses_post($input['email_template_html'] ?? $defaults['email_template_html']),
        ];

        add_settings_error('cryptnote_pro_settings', 'cryptnote_pro_settings_saved', __('Settings saved.', 'cryptnote-secure-links'), 'updated');

        return $clean;
    }

    public function render_page(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('You do not have permission to access this page.', 'cryptnote-secure-links'));
        }

        $settings = cryptnote_pro_get_settings();
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php esc_html_e('CryptNote - Secure Links', 'cryptnote-secure-links'); ?></h1>
            <p><?php esc_html_e('Generate encrypted links via CryptNote API and configure secure email sending.', 'cryptnote-secure-links'); ?></p>

            <hr class="wp-header-end">

            <div class="metabox-holder">
                <div class="postbox-container" style="min-width: 60%;">
                    <div class="postbox">
                        <div class="postbox-header">
                            <h2 class="hndle"><?php esc_html_e('API Settings and Defaults', 'cryptnote-secure-links'); ?></h2>
                        </div>
                        <div class="inside">
                            <form method="post" action="options.php">
                                <?php settings_fields('cryptnote_pro_settings_group'); ?>
                                <table class="form-table" role="presentation">
                                    <tr>
                                        <th scope="row"><label for="cryptnote_api_base"><?php esc_html_e('API Base Endpoint', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <input name="<?php echo esc_attr($this->option_key); ?>[api_base]" id="cryptnote_api_base" type="url" class="regular-text code" value="<?php echo esc_attr($settings['api_base']); ?>" required>
                                            <p class="description"><?php esc_html_e('Usually https://cryptnote.pro/api/v1', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="cryptnote_api_key"><?php esc_html_e('API Key (if required)', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <input name="<?php echo esc_attr($this->option_key); ?>[api_key]" id="cryptnote_api_key" type="text" class="regular-text" value="<?php echo esc_attr($settings['api_key']); ?>" autocomplete="off">
                                            <p class="description"><?php esc_html_e('Sent in X-API-Key header when provided.', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="cryptnote_default_max_views"><?php esc_html_e('Default Views', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <input name="<?php echo esc_attr($this->option_key); ?>[default_max_views]" id="cryptnote_default_max_views" type="number" min="1" max="20" value="<?php echo esc_attr((int)$settings['default_max_views']); ?>">
                                            <p class="description"><?php esc_html_e('Default view limit per link (1-20).', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="cryptnote_default_expire_minutes"><?php esc_html_e('Default Expiration (minutes)', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <input name="<?php echo esc_attr($this->option_key); ?>[default_expire_minutes]" id="cryptnote_default_expire_minutes" type="number" min="1" max="10080" value="<?php echo esc_attr($settings['default_expire_minutes']); ?>">
                                            <p class="description"><?php esc_html_e('Leave blank for no time expiration.', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><?php esc_html_e('Default Content Format', 'cryptnote-secure-links'); ?></th>
                                        <td>
                                            <fieldset>
                                                <label style="margin-right:16px;">
                                                    <input name="<?php echo esc_attr($this->option_key); ?>[default_format]" type="radio" value="text" <?php checked($settings['default_format'], 'text'); ?>>
                                                    <?php esc_html_e('Plain Text', 'cryptnote-secure-links'); ?>
                                                </label>
                                                <label style="margin-right:16px;">
                                                    <input name="<?php echo esc_attr($this->option_key); ?>[default_format]" type="radio" value="markdown" <?php checked($settings['default_format'], 'markdown'); ?>>
                                                    <?php esc_html_e('Markdown', 'cryptnote-secure-links'); ?>
                                                </label>
                                                <label>
                                                    <input name="<?php echo esc_attr($this->option_key); ?>[default_format]" type="radio" value="html" <?php checked($settings['default_format'], 'html'); ?>>
                                                    <?php esc_html_e('HTML', 'cryptnote-secure-links'); ?>
                                                </label>
                                            </fieldset>
                                            <p class="description"><?php esc_html_e('Defines how content will be interpreted when creating links (text, Markdown, or HTML).', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><?php esc_html_e('Enable secure links for all emails', 'cryptnote-secure-links'); ?></th>
                                        <td>
                                            <label>
                                                <input name="<?php echo esc_attr($this->option_key); ?>[email_enable]" type="checkbox" value="1" <?php checked($settings['email_enable'], 1); ?>>
                                                <?php esc_html_e('Replaces the body of emails sent by WordPress with a CryptNote link.', 'cryptnote-secure-links'); ?>
                                            </label>
                                            <p class="description"><?php esc_html_e('Attachments are removed to prevent data leakage.', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="cryptnote_email_template"><?php esc_html_e('Email Body Template', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <textarea name="<?php echo esc_attr($this->option_key); ?>[email_template]" id="cryptnote_email_template" rows="5" class="large-text code"><?php echo esc_textarea($settings['email_template']); ?></textarea>
                                            <p class="description">
                                                <?php esc_html_e('Use {share_url}, {token}, {max_views} and {expire_minutes} as placeholders.', 'cryptnote-secure-links'); ?>
                                            </p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><?php esc_html_e('Send email as HTML (clickable link)', 'cryptnote-secure-links'); ?></th>
                                        <td>
                                            <label>
                                                <input name="<?php echo esc_attr($this->option_key); ?>[email_send_html]" type="checkbox" value="1" <?php checked($settings['email_send_html'], 1); ?>>
                                                <?php esc_html_e('Sends the body as HTML so the link is clickable and compatible with HTML content.', 'cryptnote-secure-links'); ?>
                                            </label>
                                            <p class="description"><?php esc_html_e('When disabled, the body is sent as plain text.', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="cryptnote_email_template_html"><?php esc_html_e('HTML Email Template', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <textarea name="<?php echo esc_attr($this->option_key); ?>[email_template_html]" id="cryptnote_email_template_html" rows="6" class="large-text code"><?php echo esc_textarea($settings['email_template_html']); ?></textarea>
                                            <p class="description">
                                                <?php esc_html_e('Placeholders: {share_url}, {share_link} (anchor), {token}, {max_views}, {expire_minutes}.', 'cryptnote-secure-links'); ?>
                                            </p>
                                        </td>
                                    </tr>
                                </table>
                                <?php submit_button(__('Save Settings', 'cryptnote-secure-links')); ?>
                            </form>
                        </div>
                    </div>
                </div>

                <div class="postbox-container" style="min-width: 60%;">
                    <div class="postbox">
                        <div class="postbox-header">
                            <h2 class="hndle"><?php esc_html_e('Generate encrypted link now', 'cryptnote-secure-links'); ?></h2>
                        </div>
                        <div class="inside">
                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                                <?php wp_nonce_field('cryptnote_pro_generate', 'cryptnote_pro_generate_nonce'); ?>
                                <input type="hidden" name="action" value="cryptnote_pro_generate">

                                <table class="form-table" role="presentation">
                                    <tr>
                                        <th scope="row"><label for="cryptnote_content"><?php esc_html_e('Content to protect', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <textarea name="cryptnote_content" id="cryptnote_content" rows="6" class="large-text code" required></textarea>
                                            <p class="description"><?php esc_html_e('Text to be encrypted by the API.', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="cryptnote_password"><?php esc_html_e('Optional Password', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <input type="text" name="cryptnote_password" id="cryptnote_password" class="regular-text" autocomplete="off">
                                            <p class="description"><?php esc_html_e('Leave blank to not require a password.', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="cryptnote_max_views"><?php esc_html_e('Views', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <input type="number" name="cryptnote_max_views" id="cryptnote_max_views" min="1" max="20" value="<?php echo esc_attr((int)$settings['default_max_views']); ?>">
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><label for="cryptnote_expire_minutes"><?php esc_html_e('Expiration (minutes)', 'cryptnote-secure-links'); ?></label></th>
                                        <td>
                                            <input type="number" name="cryptnote_expire_minutes" id="cryptnote_expire_minutes" min="1" max="10080" value="<?php echo esc_attr($settings['default_expire_minutes']); ?>">
                                            <p class="description"><?php esc_html_e('Leave blank for no expiration.', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                    <tr>
                                        <th scope="row"><?php esc_html_e('Content Format', 'cryptnote-secure-links'); ?></th>
                                        <td>
                                            <fieldset>
                                                <label style="margin-right:12px;">
                                                    <input type="radio" name="cryptnote_format" value="text" <?php checked($settings['default_format'], 'text'); ?>>
                                                    <?php esc_html_e('Text', 'cryptnote-secure-links'); ?>
                                                </label>
                                                <label style="margin-right:12px;">
                                                    <input type="radio" name="cryptnote_format" value="markdown" <?php checked($settings['default_format'], 'markdown'); ?>>
                                                    <?php esc_html_e('Markdown', 'cryptnote-secure-links'); ?>
                                                </label>
                                                <label>
                                                    <input type="radio" name="cryptnote_format" value="html" <?php checked($settings['default_format'], 'html'); ?>>
                                                    <?php esc_html_e('HTML', 'cryptnote-secure-links'); ?>
                                                </label>
                                            </fieldset>
                                            <p class="description"><?php esc_html_e('Choose how the content should be rendered when opening the link.', 'cryptnote-secure-links'); ?></p>
                                        </td>
                                    </tr>
                                </table>
                                <?php submit_button(__('Create secure link', 'cryptnote-secure-links'), 'primary', 'cryptnote_create'); ?>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    public function handle_generate(): void
    {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Permission denied.', 'cryptnote-secure-links'));
        }

        check_admin_referer('cryptnote_pro_generate', 'cryptnote_pro_generate_nonce');

        $settings = cryptnote_pro_get_settings();
        $format = isset($_POST['cryptnote_format']) ? sanitize_text_field(wp_unslash((string)$_POST['cryptnote_format'])) : ($settings['default_format'] ?? 'text');
        if (!in_array($format, ['text', 'markdown', 'html'], true)) {
            $format = 'text';
        }

        if ($format === 'html') {
            $content = isset($_POST['cryptnote_content']) ? wp_kses_post(wp_unslash((string)$_POST['cryptnote_content'])) : '';
        } else {
            $content = isset($_POST['cryptnote_content']) ? sanitize_textarea_field(wp_unslash((string)$_POST['cryptnote_content'])) : '';
        }
        $password = isset($_POST['cryptnote_password']) ? sanitize_text_field(wp_unslash((string)$_POST['cryptnote_password'])) : '';
        $max_views_raw = isset($_POST['cryptnote_max_views']) ? sanitize_text_field(wp_unslash((string)$_POST['cryptnote_max_views'])) : '';
        $max_views = ($max_views_raw === '') ? (int)$settings['default_max_views'] : (int)$max_views_raw;
        $max_views = max(1, min(20, $max_views));

        $expire_input_raw = isset($_POST['cryptnote_expire_minutes']) ? sanitize_text_field(wp_unslash((string)$_POST['cryptnote_expire_minutes'])) : '';
        $expire_input = trim($expire_input_raw);
        $expire_minutes = $expire_input === '' ? '' : max(1, min(10080, (int)$expire_input));

        $is_markdown = $format === 'markdown';
        $is_html = $format === 'html';

        $payload = [
            'content' => $content,
            'max_views' => $max_views,
            'is_markdown' => $is_markdown,
            'is_html' => $is_html,
        ];
        if ($password !== '') {
            $payload['password'] = $password;
        }
        if ($expire_minutes !== '') {
            $payload['expire_minutes'] = $expire_minutes;
        }

        $client = new CryptNote_Client($settings);
        $result = $client->create($payload);

        if (is_wp_error($result)) {
            $this->store_notice([
                'type' => 'error',
                /* translators: %s: error message from API */
                'message' => sprintf(__('Error generating link: %s', 'cryptnote-secure-links'), $result->get_error_message()),
            ]);
        } else {
            $this->store_notice([
                'type' => 'success',
                'message' => __('Secure link created successfully.', 'cryptnote-secure-links'),
                'share_url' => $result['share_url'] ?? '',
                'token' => $result['token'] ?? '',
            ]);
        }

        wp_safe_redirect(add_query_arg(['page' => $this->page_slug], admin_url('admin.php')));
        exit;
    }

    public function render_notices(): void
    {
        $screen = get_current_screen();
        if (!$screen || $screen->id !== 'toplevel_page_' . $this->page_slug) {
            return;
        }

        settings_errors('cryptnote_pro_settings');

        $notice = $this->consume_notice();
        if (!$notice) {
            return;
        }

        $class = $notice['type'] === 'success' ? 'notice-success' : 'notice-error';
        ?>
        <div class="notice <?php echo esc_attr($class); ?> is-dismissible">
            <p><?php echo esc_html($notice['message']); ?></p>
            <?php if (!empty($notice['share_url'])) : ?>
                <p><strong><?php esc_html_e('Link:', 'cryptnote-secure-links'); ?></strong> <a href="<?php echo esc_url($notice['share_url']); ?>" target="_blank" rel="noopener noreferrer"><?php echo esc_html($notice['share_url']); ?></a></p>
                <?php if (!empty($notice['token'])) : ?>
                    <p><strong><?php esc_html_e('Token:', 'cryptnote-secure-links'); ?></strong> <code><?php echo esc_html($notice['token']); ?></code></p>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <?php
    }

    private function store_notice(array $notice): void
    {
        set_transient($this->notice_key(), $notice, 60);
    }

    private function consume_notice(): ?array
    {
        $notice = get_transient($this->notice_key());
        if ($notice) {
            delete_transient($this->notice_key());
            return (array)$notice;
        }
        return null;
    }

    private function notice_key(): string
    {
        $user = get_current_user_id();
        return 'cryptnote_pro_notice_' . ($user ?: 'guest');
    }
}
